// // Constants and configurations
const STORAGE_KEY = "selectedColors";
const DEBUG = true; // Set to false in production if needed

// Utility function for logging
function log(message, data) {
  if (DEBUG) {
    console.log(`[Color Picker] ${message}:`, data);
  }
}

document.addEventListener("DOMContentLoaded", function () {
  log("DOM Content Loaded");
  initializeColorPicker();
});

function initializeColorPicker() {
  const colorBoxes = document.querySelectorAll(".color-box");
  log("Found color boxes", colorBoxes.length);

  // Verify color boxes have proper data attributes
  colorBoxes.forEach((box, index) => {
    log(`Color box ${index}`, {
      color: box.dataset.color,
      hasDataAttribute: box.hasAttribute("data-color"),
    });
  });

  // First restore any saved state
  restoreSelectedColors();

  // Then add click handlers
  colorBoxes.forEach((box) => {
    box.addEventListener("click", handleColorSelection);
  });
}

function handleColorSelection(event) {
  const box = event.currentTarget;
  const color = box.dataset.color;

  log("Color clicked", {
    color: color,
    element: box,
  });

  // Toggle selection
  box.classList.toggle("selected");

  // Save the new state
  saveSelectedColors();
}

function saveSelectedColors() {
  try {
    const colorBoxes = document.querySelectorAll(".color-box");
    const selectedColors = [];
    let selectedCount = 0;

    colorBoxes.forEach((box) => {
      if (box.classList.contains("selected")) {
        const color = box.dataset.color;
        if (color) {
          selectedColors.push(color);
          selectedCount++;
        }
      }
    });

    log("Selected colors before save", {
      count: selectedCount,
      colors: selectedColors,
    });

    // Save to localStorage
    localStorage.setItem(STORAGE_KEY, JSON.stringify(selectedColors));

    // Verify save was successful
    const savedData = localStorage.getItem(STORAGE_KEY);
    const parsedData = JSON.parse(savedData);

    log("Save verification", {
      saved: savedData,
      parsed: parsedData,
    });

    return true;
  } catch (error) {
    console.error("Error saving colors:", error);
    return false;
  }
}

function restoreSelectedColors() {
  try {
    // Get saved data
    const savedData = localStorage.getItem(STORAGE_KEY);
    log("Retrieved from localStorage", savedData);

    if (!savedData) {
      log("No saved data found");
      return;
    }

    // Parse the saved data
    const savedColors = JSON.parse(savedData);
    log("Parsed saved colors", savedColors);

    if (!Array.isArray(savedColors)) {
      throw new Error("Saved data is not an array");
    }

    // Clear existing selections
    const colorBoxes = document.querySelectorAll(".color-box");
    colorBoxes.forEach((box) => box.classList.remove("selected"));

    // Restore selections
    let restoredCount = 0;
    colorBoxes.forEach((box) => {
      if (savedColors.includes(box.dataset.color)) {
        box.classList.add("selected");
        restoredCount++;
      }
    });

    log("Restore complete", {
      totalRestored: restoredCount,
      colors: savedColors,
    });
  } catch (error) {
    console.error("Error restoring colors:", error);
  }
}

// Add these debug utility functions
window.debugColorPicker = {
  // Check current state
  checkState: function () {
    const saved = localStorage.getItem(STORAGE_KEY);
    const selected = document.querySelectorAll(".color-box.selected");
    console.log("Current state:", {
      localStorage: saved ? JSON.parse(saved) : null,
      selectedElements: Array.from(selected).map((el) => el.dataset.color),
    });
  },

  // Force clear state
  clearState: function () {
    localStorage.removeItem(STORAGE_KEY);
    document
      .querySelectorAll(".color-box.selected")
      .forEach((el) => el.classList.remove("selected"));
    console.log("State cleared");
  },

  // Verify data attributes
  checkDataAttributes: function () {
    const boxes = document.querySelectorAll(".color-box");
    boxes.forEach((box, i) => {
      console.log(`Box ${i}:`, {
        color: box.dataset.color,
        hasAttribute: box.hasAttribute("data-color"),
        classes: box.className,
      });
    });
  },
};

// Handle page visibility changes
document.addEventListener("visibilitychange", function () {
  if (!document.hidden) {
    restoreSelectedColors();
  }
});

// Handle page show events (back/forward navigation)
window.addEventListener("pageshow", function (event) {
  if (event.persisted) {
    restoreSelectedColors();
  }
});
