<?php
// Simple SQLite helper for orders persistence
function getDbPath() {
    return __DIR__ . '/orders.sqlite';
}

function getDb()
{
    $path = getDbPath();
    $pdo = new PDO('sqlite:' . $path);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $pdo;
}

function initDb()
{
    $pdo = getDb();
    $pdo->exec("CREATE TABLE IF NOT EXISTS orders (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        order_id TEXT UNIQUE,
        amount REAL,
        package_name TEXT,
        customer_email TEXT,
        status TEXT,
        transaction_result TEXT,
        created_at TEXT,
        updated_at TEXT,
        verified_at TEXT
    )");
    // Add column if older DB schema exists
    $cols = $pdo->query("PRAGMA table_info(orders)")->fetchAll(PDO::FETCH_ASSOC);
    $hasEmail = false;
    foreach ($cols as $c) { if (isset($c['name']) && $c['name'] === 'customer_email') { $hasEmail = true; break; } }
    if (!$hasEmail) {
        try { $pdo->exec("ALTER TABLE orders ADD COLUMN customer_email TEXT"); } catch (Exception $e) { /* ignore */ }
    }
    return $pdo;
}

function insertOrder($order_id, $amount, $package_name, $customer_email = null)
{
    $pdo = initDb();
    $stmt = $pdo->prepare('INSERT OR IGNORE INTO orders (order_id, amount, package_name, customer_email, status, created_at, updated_at) VALUES (:order_id, :amount, :package_name, :customer_email, :status, :created_at, :updated_at)');
    $now = date('c');
    $stmt->execute([
        ':order_id' => $order_id,
        ':amount' => $amount,
        ':package_name' => $package_name,
        ':customer_email' => $customer_email,
        ':status' => 'requested',
        ':created_at' => $now,
        ':updated_at' => $now,
    ]);
}

function updateOrderStatus($order_id, $status, $transaction_result = null)
{
    $pdo = initDb();
    $now = date('c');
    $stmt = $pdo->prepare('UPDATE orders SET status = :status, transaction_result = :transaction_result, updated_at = :updated_at, verified_at = CASE WHEN :status = "paid" THEN :updated_at ELSE verified_at END WHERE order_id = :order_id');
    $stmt->execute([
        ':status' => $status,
        ':transaction_result' => $transaction_result,
        ':updated_at' => $now,
        ':order_id' => $order_id,
    ]);
}

function fetchOrders($limit = 100)
{
    $pdo = initDb();
    $stmt = $pdo->prepare('SELECT * FROM orders ORDER BY created_at DESC LIMIT :limit');
    $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}
