<?php
// Basic Nochex callback/notify handler with best-effort server verification
$logFile = __DIR__ . '/nochex_ipn.log';
$ordersLog = __DIR__ . '/nochex_orders.log';

$entry = "--- New callback at " . date('c') . " ---\n";
$entry .= "REMOTE_ADDR: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . "\n";
$entry .= "GET: " . print_r($_GET, true) . "\n";
$entry .= "POST: " . print_r($_POST, true) . "\n";
file_put_contents($logFile, $entry, FILE_APPEND | LOCK_EX);

// Load optional config (merchant api key, verify endpoint, etc.)
$configFile = __DIR__ . '/nochex_config.php';
$config = null;
if (file_exists($configFile)) {
	$config = include $configFile;
}

// Best-effort verification: POST the received data back to a verification endpoint.
// Use configured verify_url if present, otherwise fallback to the configured nochex URL.
$VERIFY_URL = $config['verify_url'] ?? ($config['nochex_url'] ?? 'https://secure-test.nochex.com/');

$postData = http_build_query($_POST);
$ch = curl_init($VERIFY_URL);
$options = [
	CURLOPT_RETURNTRANSFER => true,
	CURLOPT_POST => true,
	CURLOPT_POSTFIELDS => $postData,
	CURLOPT_TIMEOUT => 15,
	CURLOPT_SSL_VERIFYPEER => ($config['verify_ssl'] ?? true),
];
curl_setopt_array($ch, $options);
$verifyResponse = curl_exec($ch);
$curlErr = curl_error($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

$verifyLog = sprintf("%s | verify_response | url=%s | http=%s | err=%s | body=%s\n", date('c'), $VERIFY_URL, $httpCode, $curlErr, substr($verifyResponse,0,1000));
file_put_contents($logFile, $verifyLog, FILE_APPEND | LOCK_EX);

$status = 'pending';

// 1) Prefer explicit verification response from verification endpoint
if ($verifyResponse !== false && $httpCode >= 200 && $httpCode < 300) {
	$r = strtoupper((string)$verifyResponse);
	if (strpos($r, 'AUTHORISED') !== false || strpos($r, 'OK') !== false || strpos($r, 'AUTH') !== false) {
		$status = 'paid';
	}
}

// 2) If verify endpoint is inconclusive and an API key is configured, attempt HMAC verification
if ($status !== 'paid' && !empty($config['api_key'])) {
	$apiKey = $config['api_key'];
	$candidateFields = ['signature','hash','hmac','x_signature','x_hash'];
	$foundSig = null;
	$sigField = null;
	foreach ($candidateFields as $f) { if (!empty($_POST[$f])) { $foundSig = $_POST[$f]; $sigField = $f; break; } }
	if ($foundSig) {
		$calc = hash_hmac('sha256', $postData, $apiKey);
		if (hash_equals($calc, $foundSig)) {
			$status = 'paid';
			file_put_contents($logFile, date('c') . " | hmac_verified | field={$sigField}\n", FILE_APPEND | LOCK_EX);
		} else {
			file_put_contents($logFile, date('c') . " | hmac_mismatch | expected={$calc} | actual={$foundSig}\n", FILE_APPEND | LOCK_EX);
		}
	}
}

// 3) Last-resort: accept simulator POSTs that explicitly set transaction_result=AUTHORISED
if ($status !== 'paid') {
	$incomingResult = strtoupper($_POST['transaction_result'] ?? '');
	if ($incomingResult === 'AUTHORISED') {
		$status = 'paid';
		$simLog = sprintf("%s | simulator_verified | order=%s | incoming_result=%s\n", date('c'), $_POST['order_id'] ?? 'unknown', $incomingResult);
		file_put_contents($logFile, $simLog, FILE_APPEND | LOCK_EX);
	}
}

$orderId = $_POST['order_id'] ?? ($_POST['merchant_order_id'] ?? 'unknown');
$amount = $_POST['amount'] ?? ($_POST['transaction_amount'] ?? '');

$ordersEntry = sprintf("%s | order=%s | amount=%s | status=%s | verify_http=%s\n", date('c'), $orderId, $amount, $status, $httpCode);
file_put_contents($ordersLog, $ordersEntry, FILE_APPEND | LOCK_EX);

// Update orders DB if available
if (file_exists(__DIR__ . '/db.php')) {
	require_once __DIR__ . '/db.php';
	try {
		updateOrderStatus($orderId, $status, ($_POST['transaction_result'] ?? null));
	} catch (Exception $e) {
		file_put_contents($logFile, date('c') . " | DB update error: " . $e->getMessage() . "\n", FILE_APPEND | LOCK_EX);
	}
}

http_response_code(200);
echo 'OK';
